/////////////////////////////////////////////////////////////////////////////////

// Original obtained from ShaderToy.com
// Adapted, trivialy, for VGHD by TheEmu.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Use defines here rather than edit the body of the code.

#define iGlobalTime u_Elapsed
#define iResolution u_WindowSize

/////////////////////////////////////////////////////////////////////////////////

//// Constants
#define TIME iGlobalTime
#define PI 3.14159
#define EPSILON 1e-5
#define FLOATMAX 1e6
#define SPP 128
#define MRD 5
#define SPHERES 10
#define AMBIENT 0.1
#define SPACE 1.0

//// Data structures below
struct Ray
{
    vec3 position;
    vec3 direction;
};

struct Material
{
	vec3 emittance;
    vec3 reflectance;
    float reflectivity;
    float refractivity;
    float IOR;
};

struct Sphere
{
	vec3 position;
    float radius;
    Material material;
    int id;
};

struct Intersection
{
    vec3 position;
    vec3 normal;
    float t;
    Material material;
    Sphere sphere;
    int id;
};
    
struct Camera
{
    vec3 position;
    vec3 forward;
};

//// Scene data
Material mat_white_diffuse = Material(vec3(0.0), vec3(1.0), 0.0, 0.0, 1.0);
Material mat_red_diffuse = Material(vec3(0.0), vec3(1.0, 0.0, 0.0), 0.0, 0.0, 1.0);
Material mat_green_diffuse = Material(vec3(0.0), vec3(0.0, 1.0, 0.0), 0.0, 0.0, 1.0);
Material mat_blue_diffuse = Material(vec3(0.0), vec3(0.0, 0.0, 1.0), 0.0, 0.0, 1.0);
Material mat_mirror = Material(vec3(0.0), vec3(0.0, 0.0, 0.0), 1.0, 0.0, 1.0);
Material mat_glass_IOR105 = Material(vec3(0.0), vec3(0.0, 0.0, 0.0), 0.0, 1.0, 1.05);
Material mat_glass_IOR11 = Material(vec3(0.0), vec3(0.0, 0.0, 0.0), 0.0, 1.0, 1.1);
Material mat_glass_IOR125 = Material(vec3(0.0), vec3(0.0, 0.0, 0.0), 0.0, 1.0, 1.25);
Material mat_glass_IOR152 = Material(vec3(0.0), vec3(0.0, 0.0, 0.0), 0.0, 1.0, 1.52);
Material mat_glass_IOR175 = Material(vec3(0.0), vec3(0.0, 0.0, 0.0), 0.0, 1.0, 1.75);
Material mat_glass_IOR252 = Material(vec3(0.0), vec3(0.0, 0.0, 0.0), 0.0, 1.0, 2.52);
Material mat_glass_IOR102 = Material(vec3(0.0), vec3(0.0, 0.0, 0.0), 0.0, 1.0, 1.02);

vec3 L = vec3(0.33, 0.33, 0.33);
Camera camera = Camera(vec3(0.0, 0.0, 2.5), vec3(0.0, 0.0, -1.0));

Sphere scene_spheres[SPHERES];

//// Initialize the scene
void initScene()
{
    scene_spheres[0] = Sphere(vec3(2.5 * cos(TIME), 2.5 * sin(TIME), -5.0), 1.0, mat_red_diffuse, -1);
    scene_spheres[1] = Sphere(vec3(2.5 * cos(TIME * 1.25), 0.0, -5.0 + 2.5 * sin(TIME * 1.25)), 1.0, mat_green_diffuse, -1);
    scene_spheres[2] = Sphere(vec3(0.0, 0.0, -5.0), 1.0, mat_white_diffuse, -1);
    scene_spheres[3] = Sphere(vec3(5.0 * cos(TIME * 0.3), 0.0, -5.0 + 5.0 * sin(TIME * 0.3)), 0.75, mat_glass_IOR105, -1);
    scene_spheres[4] = Sphere(vec3(5.0 * cos(TIME * 0.3 + PI * 0.25), 0.5 * sin(TIME * 1.0), -5.0 + 5.0 * sin(TIME * 0.3 + PI * 0.25)), 0.5, mat_glass_IOR11, -1);
    scene_spheres[5] = Sphere(vec3(5.0 * cos(TIME * 0.3 + PI * 0.5), 0.5 * sin(TIME * 0.5), -5.0 + 5.0 * sin(TIME * 0.3 + PI * 0.5)), 0.625, mat_glass_IOR125, -1);
    scene_spheres[6] = Sphere(vec3(5.0 * cos(TIME * 0.3 + PI * 0.75), 0.5 * sin(TIME * 0.25), -5.0 + 5.0 * sin(TIME * 0.3 + PI * 0.75)), 0.6, mat_glass_IOR152, -1);
	scene_spheres[7] = Sphere(vec3(5.0 * cos(TIME * 0.3 + PI * 1.0), 0.5 * sin(TIME * 0.25), -5.0 + 5.0 * sin(TIME * 0.3 + PI * 1.0)), 0.6, mat_glass_IOR175, -1);
    scene_spheres[8] = Sphere(vec3(5.0 * cos(TIME * 0.3 + PI * 1.25), 0.5 * sin(TIME * 0.25), -5.0 + 5.0 * sin(TIME * 0.3 + PI * 1.25)), 0.6, mat_glass_IOR252, -1);
    scene_spheres[9] = Sphere(vec3(5.0 * cos(TIME * 0.3 + PI * 1.5), 0.5 * sin(TIME * 0.25), -5.0 + 5.0 * sin(TIME * 0.3 + PI * 1.5)), 0.6, mat_glass_IOR102, -1);
}

//// Ray -> Sphere intersection
bool intersectSphere(in Ray r, in Sphere s, inout Intersection x)
{
    if (x.id == s.id)
        return false;
    
    vec3 op = s.position - r.position;
    float t;
    float b = dot(op, r.direction);
    float det = b * b - dot(op, op) + s.radius * s.radius;
    
    if (det < 0.0)
        return false;
    
    det = sqrt(det);
    t = (t = b - det) > EPSILON ? t : ((t = b + det) > EPSILON ? t : -1.0);
    
    if (t == -1.0 || t > x.t)
        return false;
    
    x.position = r.position + r.direction * t;
    x.normal = (x.position - s.position) / s.radius;
    x.t = t;
    x.material = s.material;
    x.id = s.id;
    
    return true;
}

//// Method to test ray intersection with all scene objects
void iScene(in Ray r, inout Intersection x)
{
	// Intersect scene spheres
    for (int i = 0; i < SPHERES; i++)
    {
        scene_spheres[i].id = i;
        intersectSphere(r, scene_spheres[i], x);
    }
}

//// Raytracing
vec3 trace(inout Ray r, inout Intersection iSection)
{
    vec3 color_final = vec3(0.0);
    
    iSection.t = FLOATMAX;
    iSection.id = 128;
    
    // Recursively trace the scene
    for (int i = 0; i < MRD; i++)
    {
        // Intersect the ray against all scene primitives
        iScene(r, iSection);
        
        // Break out from the loop if no intersection happened
        if (iSection.t >= FLOATMAX - 1.0) // See explanation for -1.0 at the end of this for loop
            break;
        
        // Store some variables into temp ones with shorter names
        vec3 RO = r.position;
        vec3 RD = r.direction;
        vec3 I = iSection.position;
        vec3 N = iSection.normal;
        Material M = iSection.material;
        
        // Lambertian shading
        if (length(M.reflectance) > 0.0)
        {
            color_final += M.reflectance * dot(N, L);
        }
        
        // Quit the loop if no recursion is required
        if (M.reflectivity <= 0.0 && M.refractivity <= 0.0)
            break;
        
        // Reflect the ray if surface is a mirror
        if (M.reflectivity > 0.0)
        {
            r.position = iSection.position;
            r.direction = normalize(reflect(RD, N));
        }
        
        // Refract the ray if surface is a dielectric material
        if (M.refractivity > 0.0)
        {
            float NdotI = dot(RD, N), n, n1, n2;
           	
            // Are we going into the medium or getting out from it
            if (NdotI > 0.0)
            {
                n2 = SPACE;
                n1 = M.IOR;
                N = -N;
            } else {
                n2 = M.IOR;
                n1 = SPACE;
                NdotI = -NdotI;
            }
            
            // Calculate correct index of refraction, corrected n1 & n2 which was pointed out by user valentingalea
            n = n1 / n2;
            
            r.position = iSection.position;
            r.direction = normalize(refract(RD, N, n));
        }
        
        // Reset intersection data for a new round
        iSection.t = FLOATMAX - 1.0; // -1.0, a simple trick to identify if the ray has gone through recursion or not
        iSection.material.reflectivity = 0.0;
        iSection.material.refractivity = 0.0;
    }
    
    // Add the ambient term if we hit something
    if (iSection.t <= FLOATMAX - 1.0)
        color_final += AMBIENT;
    
    return color_final;
}

void main(void)
{
    // Center the normalized coordinates
	vec2 uv = vec2(-0.5) + (gl_FragCoord.xy / iResolution.xy);
    // Fix the aspect ratio
    uv.x *= iResolution.x / iResolution.y;
    // Init camera -- ToDo: Quaternion camera for the lulz
    // Initialize scene
    initScene();
    // Calculate ray & intersection data
    Ray ray = Ray(vec3(camera.position), normalize(vec3(uv, -1.0)));
    Intersection iSection;
    // Do the path tracing
    vec3 shaded_color = trace(ray, iSection);
    gl_FragColor = vec4(clamp(shaded_color, 0.0, 1.0), 1.0);
}